<?php
/**
 * NOTICE OF LICENSE
 *
 * This file is licenced under the Software License Agreement.
 * With the purchase or the installation of the software in your application
 * you accept the licence agreement.
 *
 * You must not modify, adapt or create derivative works of this source code
 *
 * @author    D3 Data Development
 * @copyright 2016 D3 Data Development
 * @license   LICENSE.txt
 */

use d3\heidelpay\Ngw;
use d3\heidelpay\ngw\Request;
use d3\heidelpay\payment\Ideal;
use d3\heidelpay\request\Config;
use d3\heidelpay\transactionlog\reader\Transaction;

require_once(dirname(__FILE__) . '/heidelpay.php'); // Base Controller

/**
 * Class HeidelpayIdealModuleFrontController
 */
class HeidelpayIdealModuleFrontController extends HeidelpayModuleFrontController
{
    public $currentTemplate = 'ideal.tpl';

    public function initContent()
    {
        $errors = array();

        if (isset($_REQUEST['brand']) && false == empty($_REQUEST['brand'])) {
            $data = $this->getDataIdeal($_REQUEST['brand']);

            $response = $this->sendRequest($data);

            if (isset($response['PROCESSING_RESULT']) && 'ACK' === $response['PROCESSING_RESULT']) {
                $transaction = new Transaction();
                $transaction->read($response);
                $redirectUrl = $transaction->getRedirecturl();

                if (false == empty($redirectUrl)) {
                    if (false == substr_count($redirectUrl, "?")) {
                        $redirectUrl .= "?";
                    }

                    foreach ($transaction->getRedirectparameters() as $index => $value) {
                        $redirectUrl .= $index . "=" . rawurlencode($value) . '&';
                    }
                    Tools::redirectLink($redirectUrl);
                }
            } else {
                $translation = $this->module->l($response['PROCESSING_RETURN_CODE']);
                if ($translation === $response['PROCESSING_RETURN_CODE']) {
                    $translation = $this->module->l(
                        'An error occurred while processing payment code:' . $translation//
                        . '::' . __LINE__ . '::' . __CLASS__
                    );
                }
                $errors[] = $translation;
            }
        }

        $data     = $this->getDataIdealRg();
        $result = $this->validateCustomerData($data);

        if (false == empty($result)) {
            $this->errors = $result;
            $this->setTemplate("error.tpl");
            parent::initContent();

            return;
        }

        $response = $this->sendRequest($data);

        if (isset($response['PROCESSING_RESULT']) && 'ACK' === $response['PROCESSING_RESULT']) {
            $this->context->smarty->assign('brands', $this->getHeidelpayNgwBrands($response));
        } else {
            $translation = $this->module->l($response['PROCESSING_RETURN_CODE']);
            if ($translation === $response['PROCESSING_RETURN_CODE']) {
                $translation = $this->module->l(
                    'An error occurred while processing payment code:' . $translation//
                    . '::' . __LINE__ . '::' . __CLASS__
                );
            }
            $errors[] = $translation;
        }
        $this->errors = $errors;
        $this->context->smarty->assign('error_tpl', $this->module->getTemplatePath("error.tpl"));

        parent::initContent();
    }

    protected function getDataIdeal($bank)
    {
        $ideal           = new Ideal();
        $billing_address = new Address($this->context->cart->id_address_invoice);

        $config = new Config();
        $config->setIsTestmode(!Configuration::get('HEIDELPAY_LIVE_MODE'));

        $config->setResponseUrl(
            $this->context->link->getModuleLink('heidelpay', 'response')
        );

        $data                                 = array();
        $data['REQUEST.VERSION']              = "1.0";
        $data['SECURITY.SENDER']              = Configuration::get('HEIDELPAY_ACCOUNT_SECURITYSENDER');
        $data['FRONTEND.RESPONSE_URL']        = $config->getResponseUrl();
        $data['IDENTIFICATION.TRANSACTIONID'] = rand() . "__@@" . date('Ymd-Hms');
        $data['USER.LOGIN']                   = Configuration::get('HEIDELPAY_ACCOUNT_LOGIN');
        $data['USER.PWD']                     = Configuration::get('HEIDELPAY_ACCOUNT_PASSWORD');
        $data['TRANSACTION.MODE']             = $config->getTransactionMode();
        $data['TRANSACTION.RESPONSE']         = 'SYNC';
        $data['TRANSACTION.CHANNEL']          = Configuration::get('HEIDELPAY_ACCOUNT_CHANNEL_IDEAL');
        $data['PAYMENT.CODE']                 = $ideal->getPaymentCode() . ".PA";
        $data['PRESENTATION.AMOUNT']          = $this->context->cart->getOrderTotal();
        $data['PRESENTATION.CURRENCY']        = $this->context->currency->iso_code;
        $data['NAME.GIVEN']                   = $this->context->customer->firstname;
        $data['NAME.FAMILY']                  = $this->context->customer->lastname;
        $data['NAME.COMPANY']                 = $this->context->customer->company;
        $data['CONTACT.EMAIL']                = $this->context->customer->email;
        $data['CONTACT.IP']                   = Tools::getRemoteAddr();
        $data['ACCOUNT.COUNTRY']              = $this->context->country->iso_code;
        $data['ACCOUNT.BANKNAME']             = $bank;
        $data['ADDRESS.STREET']               = trim("{$billing_address->address1} {$billing_address->address2}");
        $data['ADDRESS.ZIP']                  = $billing_address->postcode;
        $data['ADDRESS.CITY']                 = $billing_address->city;
        $data['SHOP.TYPE']                    = "PrestaShop " . _PS_VERSION_;
        $data['SHOPMODULE.VERSION']           = $this->module->version;
        $data['CRITERION.SESSION']            = "ideal__@@" . $this->context->cookie->checksum//
            . '__@@' . $this->context->customer->id;
        $data['CRITERION.HASH']               = $this->generateHash($data);

        return $data;
    }

    /**
     * @param $data
     *
     * @return array
     */
    public function sendRequest($data)
    {
        $request = new Request();

        $request->setUrl($this->getHeidelpayUrl());
        $request->setCurlTimeOut($this->getHeidelpayCurlTimeOut());
        $request->setData($data);

        PrestaShopLogger::addLog(
            'HeidelpayResponse::initContent::NGW-request data: ' . var_export($data, true),
            1,
            null,
            'Heidelpay Module',
            (int)$this->context->cart->id,
            true
        );

        $ngw      = new Ngw();
        $response = $ngw->sendPostRequest($request);

        PrestaShopLogger::addLog(
            'HeidelpayResponse::initContent::NGW-response data: ' . var_export($response, true),
            1,
            null,
            'Heidelpay Module',
            (int)$this->context->cart->id,
            true
        );

        return $response;
    }

    protected function getDataIdealRg()
    {
        $ideal           = new Ideal();
        $billing_address = new Address($this->context->cart->id_address_invoice);

        $config = new Config();
        $config->setIsTestmode(!Configuration::get('HEIDELPAY_LIVE_MODE'));

        $config->setResponseUrl(
            $this->context->link->getModuleLink('heidelpay', 'response')
        );

        $data                                 = array();
        $data['REQUEST.VERSION']              = "1.0";
        $data['SECURITY.SENDER']              = Configuration::get('HEIDELPAY_ACCOUNT_SECURITYSENDER');
        $data['FRONTEND.ENABLED']             = 'true';
        $data['FRONTEND.MODE']                = 'WHITELABEL';
        $data['RISKMANAGEMENT.PROCESS']       = 'AUTO';
        $data['FRONTEND.RESPONSE_URL']        = $config->getResponseUrl();
        $data['IDENTIFICATION.TRANSACTIONID'] = rand() . "__@@" . date('Ymd-Hms');
        $data['USER.LOGIN']                   = Configuration::get('HEIDELPAY_ACCOUNT_LOGIN');
        $data['USER.PWD']                     = Configuration::get('HEIDELPAY_ACCOUNT_PASSWORD');
        $data['TRANSACTION.MODE']             = $config->getTransactionMode();
        $data['TRANSACTION.RESPONSE']         = 'SYNC';
        $data['TRANSACTION.CHANNEL']          = Configuration::get('HEIDELPAY_ACCOUNT_CHANNEL_IDEAL');
        $data['PAYMENT.CODE']                 = $ideal->getPaymentCode() . ".RG";
        $data['NAME.GIVEN']                   = $this->context->customer->firstname;
        $data['NAME.FAMILY']                  = $this->context->customer->lastname;
        $data['NAME.COMPANY']                 = $this->context->customer->company;
        $data['NAME.BIRTHDATE']               = $this->context->customer->birthday;
        $data['CONTACT.EMAIL']                = $this->context->customer->email;
        $data['CONTACT.IP']                   = Tools::getRemoteAddr();
        $data['ACCOUNT.COUNTRY']              = $this->context->country->iso_code;
        $data['ADDRESS.STREET']               = trim("{$billing_address->address1} {$billing_address->address2}");
        $data['ADDRESS.ZIP']                  = $billing_address->postcode;
        $data['ADDRESS.CITY']                 = $billing_address->city;
        $data['SHOP.TYPE']                    = "PrestaShop " . _PS_VERSION_;
        $data['SHOPMODULE.VERSION']           = $this->module->version;
        $data['CRITERION.SESSION']            = "ideal__@@" . $this->context->cookie->checksum . '__@@' //
            . $this->context->customer->id;
        $data['CRITERION.HASH']               = $this->generateHash($data);

        return $data;
    }
}
