<?php
/**
 * This Software is the property of Data Development and is protected
 * by copyright law - it is NOT Freeware.
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 * http://www.shopmodule.com
 *
 * @copyright (C) D3 Data Development (Inh. Thomas Dartsch)
 * @author        D3 Data Development <support@shopmodule.com>
 * @link          http://www.oxidmodule.com
 */

namespace d3\heidelpay\payment;

use d3\heidelpay\payment\data\Position;
use d3\heidelpay\request\exception\NoPaymentChannelException;
use d3\heidelpay\request\exception\NoPaymentMethodException;
use d3\heidelpay\request\exception\NoPaymentOptionException;

/**
 * Class Data
 *
 * @package d3\heidelpay\payment
 */
class Data
{
    /** @var  string */
    protected $referenceId;

    /** @var  string */
    protected $paymentMethod;

    /** @var  string */
    protected $paymentOption;

    /** @var  double */
    protected $amount = 0.00;

    /** @var  string */
    protected $currency;

    /** @var  string */
    protected $sessionId;

    /** @var  string */
    protected $bankName;

    /** @var  string */
    protected $accountCountry;

    /** @var  string */
    protected $accountHolder;

    /** @var bool */
    protected $isSepaMode = true;

    /** @var  string */
    protected $iban;

    /** @var  string */
    protected $bic;

    /** @var  string */
    protected $accountNumber;

    /** @var  string */
    protected $bankCode;

    /** @var  string */
    protected $transactionId;

    /** @var  string */
    protected $uniqueId;

    /** @var  string */
    protected $paymentChannel;

    /** @var  string */
    protected $userZip;

    /** @var  string */
    protected $userFamilyName;

    /** @var  string */
    protected $userFirstName;

    /** @var  string */
    protected $userCompany;

    /** @var  string */
    protected $userBirthdate = "";

    /** @var  string */
    protected $userEMailAddress;

    /** @var  string */
    protected $userCity;

    /** @var  string */
    protected $userCountryCode;

    /** @var  string */
    protected $userStreet;

    /** @var  array */
    protected $criterionPositions = array();

    /** @var  string */
    protected $userPhone;

    /** @var  string */
    protected $searchDate;

    /**
     * @param AbstractPayment $payment
     * @param array           $properties
     */
    public function __construct(AbstractPayment $payment, array $properties = array())
    {
        foreach ($properties as $propertyName => $propertyvalue) {
            $this->{$propertyName} = $propertyvalue;
        }

        $this->payment       = $payment;
        $this->paymentMethod = $payment->getPaymentCode();
    }

    /**
     * @return string
     */
    public function getTransactionId()
    {
        return $this->transactionId;
    }

    /**
     * @param string $transactionId
     */
    public function setTransactionId($transactionId)
    {
        $this->transactionId = $transactionId;
    }

    /**
     * @return string
     */
    public function getReferenceId()
    {
        return $this->referenceId;
    }

    /**
     * @param string $referenceId
     */
    public function setReferenceId($referenceId)
    {
        $this->referenceId = $referenceId;
    }

    /**
     * returns the payment combination of method and option.
     *
     * @param string $separator
     *
     * @return string
     * @throws NoPaymentMethodException
     * @throws NoPaymentOptionException
     */
    public function getPaymentCombination($separator = '.')
    {
        if (false == $this->paymentMethod) {
            $oEx = new NoPaymentMethodException();
            throw $oEx;
        }

        if (false == $this->paymentOption) {
            $oEx = new NoPaymentOptionException();
            throw $oEx;
        }

        return "{$this->paymentMethod}{$separator}{$this->paymentOption}";
    }

    /**
     * @param AbstractPayment $payment
     */
    public function setPayment($payment)
    {
        $this->payment = $payment;
    }

    /**
     * @param string $paymentMethod
     */
    public function setPaymentMethod($paymentMethod)
    {
        $this->paymentMethod = $paymentMethod;
    }

    /**
     * @param string $paymentOption
     */
    public function setPaymentOption($paymentOption)
    {
        $this->paymentOption = $paymentOption;
    }

    /**
     * @param boolean $isSepaMode
     */
    public function setIsSepaMode($isSepaMode)
    {
        $this->isSepaMode = $isSepaMode;
    }

    /**
     * @return float
     */
    public function getAmount()
    {
        return (double)$this->amount;
    }

    /**
     * @param float $amount
     */
    public function setAmount($amount)
    {
        $this->amount = $amount;
    }

    /**
     * @return string
     * @throws \Exception
     */
    public function getCurrency()
    {
        $currency = $this->currency;
        if (false == $currency) {
            throw new \Exception('no currrency is set');
        }

        return $this->currency;
    }

    /**
     * @param string $currency
     */
    public function setCurrency($currency)
    {
        $this->currency = $currency;
    }

    /**
     * @return string
     */
    public function getSessionId()
    {
        return $this->sessionId;
    }

    /**
     * @param string $sessionId
     */
    public function setSessionId($sessionId)
    {
        $this->sessionId = $sessionId;
    }

    /**
     * @return string
     */
    public function getBankName()
    {
        return $this->bankName;
    }

    /**
     * @param string $bankName
     */
    public function setBankName($bankName)
    {
        $this->bankName = $bankName;
    }

    /**
     * returns ISO Alpha 2 of a country
     *
     * @return string
     */
    public function getAccountCountry()
    {
        return $this->accountCountry;
    }

    /**
     * @param string $accountCountry
     */
    public function setAccountCountry($accountCountry)
    {
        $this->accountCountry = $accountCountry;
    }

    /**
     * @return string
     */
    public function getAccountHolder()
    {
        return $this->accountHolder;
    }

    /**
     * @param string $accountHolder
     */
    public function setAccountHolder($accountHolder)
    {
        $this->accountHolder = $accountHolder;
    }

    /**
     * @return string
     */
    public function getAccountNumber()
    {
        return $this->accountNumber;
    }

    /**
     * @param string $accountNumber
     */
    public function setAccountNumber($accountNumber)
    {
        $this->accountNumber = $accountNumber;
    }

    /**
     * @return string
     */
    public function getBankCode()
    {
        return $this->bankCode;
    }

    /**
     * @param string $bankCode
     */
    public function setBankCode($bankCode)
    {
        $this->bankCode = $bankCode;
    }

    /**
     * @return bool
     */
    public function isSepaMode()
    {
        return $this->isSepaMode;
    }

    /**
     * @return string
     */
    public function getIban()
    {
        return $this->iban;
    }

    /**
     * @param string $iban
     */
    public function setIban($iban)
    {
        $this->iban = $iban;
    }

    /**
     * @return string
     */
    public function getBic()
    {
        return $this->bic;
    }

    /**
     * @param string $bic
     */
    public function setBic($bic)
    {
        $this->bic = $bic;
    }

    /**
     * @return string
     */
    public function getUniqueId()
    {
        return $this->uniqueId;
    }

    /**
     * @param string $uniqueId
     */
    public function setUniqueId($uniqueId)
    {
        $this->uniqueId = $uniqueId;
    }

    /**
     * @return string
     * @throws NoPaymentChannelException
     */
    public function getPaymentChannel()
    {
        if (empty($this->paymentChannel)) {
            $oEx = new NoPaymentChannelException();
            throw $oEx;
        }

        return $this->paymentChannel;
    }

    /**
     * @param string $paymentChannel
     */
    public function setPaymentChannel($paymentChannel)
    {
        $this->paymentChannel = $paymentChannel;
    }

    /**
     * @return string
     */
    public function getUserFamilyName()
    {
        return $this->userFamilyName;
    }

    /**
     * @param string $userFamilyName
     */
    public function setUserFamilyName($userFamilyName)
    {
        $this->userFamilyName = $userFamilyName;
    }

    /**
     * @return string
     */
    public function getUserFirstName()
    {
        return $this->userFirstName;
    }

    /**
     * @param string $userFirstName
     */
    public function setUserFirstName($userFirstName)
    {
        $this->userFirstName = $userFirstName;
    }

    /**
     * @return string
     */
    public function getUserCompany()
    {
        return $this->userCompany;
    }

    /**
     * @param string $userCompany
     */
    public function setUserCompany($userCompany)
    {
        $this->userCompany = $userCompany;
    }

    /**
     * @return string
     */
    public function getUserBirthdate()
    {
        return $this->userBirthdate;
    }

    /**
     * @param string $userBirthdate
     */
    public function setUserBirthdate($userBirthdate)
    {
        $this->userBirthdate = $userBirthdate;
    }

    /**
     * @return string
     */
    public function getUserEMailAddress()
    {
        return $this->userEMailAddress;
    }

    /**
     * @param string $userEMailAddress
     */
    public function setUserEMailAddress($userEMailAddress)
    {
        $this->userEMailAddress = $userEMailAddress;
    }

    /**
     * @return string
     */
    public function getUserCity()
    {
        return $this->userCity;
    }

    /**
     * @param string $userCity
     */
    public function setUserCity($userCity)
    {
        $this->userCity = $userCity;
    }

    /**
     * @return string
     */
    public function getUserCountryCode()
    {
        return $this->userCountryCode;
    }

    /**
     * @param string $userCountryCode
     */
    public function setUserCountryCode($userCountryCode)
    {
        $this->userCountryCode = $userCountryCode;
    }

    /**
     * @return string
     */
    public function getUserStreet()
    {
        return $this->userStreet;
    }

    /**
     * @param string $userStreet
     */
    public function setUserStreet($userStreet)
    {
        $this->userStreet = $userStreet;
    }

    /**
     * @return string
     */
    public function getUserZip()
    {
        return $this->userZip;
    }

    /**
     * @param string $userZip
     */
    public function setUserZip($userZip)
    {
        $this->userZip = $userZip;
    }

    /**
     * @return array
     */
    public function getCriterionPositions()
    {
        return $this->criterionPositions;
    }

    /**
     * @param array $positions
     */
    public function setCriterionPositions(array $positions)
    {
        $this->criterionPositions = $positions;
    }

    /**
     * @param Position $position
     */
    public function setCriterionPosition(Position $position)
    {
        $this->criterionPositions[] = $position;
    }

    /**
     * @return string
     */
    public function getUserPhone()
    {
        return $this->userPhone;
    }

    /**
     * @param string $userPhone
     */
    public function setUserPhone($userPhone)
    {
        $this->userPhone = $userPhone;
    }

    public function getSearchDate()
    {
        return $this->searchDate;
    }

    /**
     * @param string $searchDate
     */
    public function setSearchDate($searchDate)
    {
        $this->searchDate = $searchDate;
    }
}
