<?php
/**
 * NOTICE OF LICENSE
 * This file is licenced under the Software License Agreement.
 * With the purchase or the installation of the software in your application
 * you accept the licence agreement.
 * You must not modify, adapt or create derivative works of this source code
 *
 * @author    D3 Data Development
 * @copyright 2017 D3 Data Development
 * @license   LICENSE.txt
 */

use d3\heidelpay\blowfish\Blowfish;
use d3\heidelpay\Ngw;
use d3\heidelpay\ngw\Request;
use d3\heidelpay\payment\Prepayment;
use d3\heidelpay\request\Config;

//use d3\heidelpay\transactionlog\reader\Transaction;

require_once(dirname(__FILE__).'/heidelpay.php'); // Base Controller

/**
 * Class HeidelpayPrepaymentModuleFrontController
 */
class HeidelpayPrepaymentModuleFrontController extends HeidelpayModuleFrontController
{

    /**
     * @throws PrestaShopDatabaseException
     * @throws PrestaShopException
     */
    public function initContent()
    {
        $data   = $this->getDataPrepayment();
        $result = $this->validateCustomerData($data);

        if (false == empty($result)) {
            $this->errors = $result;
            $this->setTemplate("module:heidelpay/views/templates/front/errorpage.tpl");
            parent::initContent();

            return;
        }

        $response = $this->sendRequest($data);

        if (isset($response['PROCESSING_RESULT']) && 'ACK' === $response['PROCESSING_RESULT']) {
            if (isset($response['CRITERION_SESSION'])) {
                $aTmp                                 = explode("__@@", $response['CRITERION_SESSION']);
                $response['heidelpay_input_position'] = $aTmp[0];
                $response['id_customer']              = $aTmp[2];
            }

            $blowFish = new Blowfish(HeidelpayModuleFrontController::BLOFISH_KEY);

            $activeRecord                   = new HeidelpayTransactionModel();
            $activeRecord->id_transaction   = $response['IDENTIFICATION_UNIQUEID'];
            $activeRecord->id_customer      = $response['id_customer'];
            $activeRecord->transaction_data = $blowFish->encrypt($this->encode($response));
            if (false == $activeRecord->add()) {
                PrestaShopLogger::addLog(
                    __CLASS__ . ':: coult not save transaction to database',
                    2,
                    null,
                    'Heidelpay Module',
                    (int)$this->context->cart->id,
                    true
                );
            }

            $currentTransactionId = $activeRecord->id;

            Tools::redirect(
                $this->context->link->getModuleLink(
                    'heidelpay',
                    'validation',
                    array('currenttransactionid' => $currentTransactionId)
                )
            );
        } else {
            $message = '';
            if (isset($response['PROCESSING_RETURN_CODE'])) {
                $message .= $response['PROCESSING_RETURN_CODE'].'::';
            }

            if (isset($response['PROCESSING_RETURN'])) {
                $message .= $response['PROCESSING_RETURN'];
            }
            $this->setErrorTemplate($message);
        }

        parent::initContent();
    }

    /**
     * @return array
     * @throws PrestaShopDatabaseException
     * @throws PrestaShopException
     * @throws Exception
     */
    protected function getDataPrepayment()
    {
        $prepayment      = new Prepayment();
        $billing_address = new Address($this->context->cart->id_address_invoice);
        $billing_country = new Country($billing_address->id_country);

        $config = new Config();
        $config->setIsTestmode(false == Configuration::get('HEIDELPAY_LIVE_MODE'));

        $config->setResponseUrl($this->context->link->getModuleLink('heidelpay', 'response'));
        if ($config->isTestmode()) {
            //CRITERION.PUSH_URL works only on testsystem
            $config->setNotifyUrl($this->context->link->getModuleLink('heidelpay', 'notify'));
        }

        $data                                 = array();
        $data['REQUEST.VERSION']              = "1.0";
        $data['SECURITY.SENDER']              = Configuration::get('HEIDELPAY_ACCOUNT_SECURITYSENDER');
        $data['FRONTEND.RESPONSE_URL']        = $config->getResponseUrl();
        $data['IDENTIFICATION.TRANSACTIONID'] = rand()."__@@".date('Ymd-Hms');
        $data['USER.LOGIN']                   = Configuration::get('HEIDELPAY_ACCOUNT_LOGIN');
        $data['USER.PWD']                     = Configuration::get('HEIDELPAY_ACCOUNT_PASSWORD');
        $data['TRANSACTION.MODE']             = $config->getTransactionMode();
        $data['TRANSACTION.RESPONSE']         = 'SYNC';
        $data['TRANSACTION.CHANNEL']          = Configuration::get('HEIDELPAY_ACCOUNT_CHANNEL_PREPAYMENT');
        $data['PAYMENT.CODE']                 = $prepayment->getPaymentCode().".PA";
        $data['PRESENTATION.AMOUNT']          = $this->context->cart->getOrderTotal();
        $data['PRESENTATION.CURRENCY']        = $this->context->currency->iso_code;
        $data['NAME.GIVEN']                   = $this->context->customer->firstname;
        $data['NAME.FAMILY']                  = $this->context->customer->lastname;
        $data['NAME.COMPANY']                 = $this->context->customer->company;
        $data['NAME.BIRTHDATE']               = $this->context->customer->birthday;
        $data['CONTACT.EMAIL']                = $this->context->customer->email;
        $data['CONTACT.IP']                   = Tools::getRemoteAddr();
        $data['ADDRESS.COUNTRY']              = $billing_country->iso_code;
        $data['ADDRESS.STREET']               = trim("{$billing_address->address1} {$billing_address->address2}");
        $data['ADDRESS.ZIP']                  = $billing_address->postcode;
        $data['ADDRESS.CITY']                 = $billing_address->city;
        $data['SHOP.TYPE']                    = "PrestaShop "._PS_VERSION_;
        $data['SHOPMODULE.VERSION']           = $this->module->version;
        $checksum                             = $this->context->cookie->getFamily('checksum')['checksum'];
        $data['CRITERION.SESSION']            = "prepayment__@@".$checksum.'__@@'.$this->context->customer->id;
        $data['CRITERION.HASH']               = $this->generateHash($data);
        if ($config->isTestmode()) {
            //CRITERION.PUSH_URL works only on testsystem
            $data['CRITERION.PUSH_URL'] = $config->getNotifyUrl();
        }

        return $data;
    }

    /**
     * @param $data
     *
     * @return array
     */
    public function sendRequest($data)
    {
        $request = new Request();

        $request->setUrl($this->getHeidelpayUrl());
        $request->setCurlTimeOut($this->getHeidelpayCurlTimeOut());
        $request->setData($data);

        PrestaShopLogger::addLog(
            'HeidelpayResponse::initContent::NGW-request data: ' . var_export($data, true),
            1,
            null,
            'Heidelpay Module',
            (int)$this->context->cart->id,
            true
        );

        $ngw      = new Ngw();
        $response = $ngw->sendPostRequest($request);

        PrestaShopLogger::addLog(
            'HeidelpayResponse::initContent::NGW-response data: ' . var_export($response, true),
            1,
            null,
            'Heidelpay Module',
            (int)$this->context->cart->id,
            true
        );

        return $response;
    }
}
