<?php

namespace d3\heidelpay\transactionlog\reader;

use DOMDocument;
use Domxpath;

/**
 * Class Transaction
 */
class Transaction
{
    /** @var string */
    public $result;

    /** @var string */
    public $status;

    /** @var string */
    public $reason;

    /** @var string */
    public $timestamp;

    /** @var string */
    public $amount;

    /** @var string */
    public $currency;

    /** @var string */
    public $paymentcode;

    /** @var string */
    public $transactionid;

    /** @var string */
    public $uniqueid;

    /** @var string */
    public $shortid;

    /** @var string */
    public $reasoncode;

    /** @var string */
    public $authenticationtype;

    /** @var string */
    public $returncode;

    /** @var string */
    public $accountbank;

    /** @var string */
    public $accountholder;

    /** @var string */
    public $accountbrand;

    /** @var string */
    public $accountmonth;

    /** @var string */
    public $accountyear;

    /** @var string */
    public $redirecturl;

    /** @var array */
    public $redirectparameters = array();

    /** @var string */
    public $statuscode;

    /** @var string */
    public $inputposition;

    /** @var string */
    public $accountnumber;

    public $return;

    /** @var string */
    protected $basketId;

    /** @var string */
    protected $method;

    /** @var array */
    protected $basketErrors = array();

    /**
     * read transaction data and fill properties
     *
     * @param $mTransactionData
     *
     * @return Transaction
     */
    public function read($mTransactionData)
    {
        $this->transactionid = '';
        $this->uniqueid      = '';
        $this->shortid       = '';
        $this->result        = '';
        $this->status        = '';
        $this->reason        = '';
        $this->timestamp     = '';
        $this->amount        = '';
        $this->currency      = '';
        $this->paymentcode   = '';

        if ($this->isValidJSON($mTransactionData)) {
            $this->readJson(json_decode($mTransactionData));

            return $this;
        }

        $oldSetting = libxml_use_internal_errors(true);
        $xml        = new DOMDocument();

        if (is_array($mTransactionData)) {
            if (isset($mTransactionData['response'])) {
                $xml->loadXML(str_replace("'", '', urldecode($mTransactionData['response'])));
                $oDomXPath = new Domxpath($xml);

                $this->readXmlData($oDomXPath, $mTransactionData);
            } else {
                $this->readArray($mTransactionData);
            }
        } elseif ($xml->loadXML($mTransactionData)) {
            $oDomXPath = new Domxpath($xml);
            $this->readXmlData($oDomXPath);
        }

        libxml_clear_errors();
        libxml_use_internal_errors($oldSetting);

        return $this;
    }

    /**
     * @param $string
     *
     * @return bool
     */
    public function isValidJSON($string)
    {
        if (is_null($string) || false == is_string($string)) {
            return false;
        }

        if (is_int($string) || is_float($string)) {
            return true;
        }

        @json_decode($string);

        return json_last_error() === JSON_ERROR_NONE;
    }

    /**
     * @param $json
     */
    protected function readJson($json)
    {
        $this->readJsonProp($json, "result");
        $this->readJsonProp($json, "basketId");
        $this->readJsonProp($json, "method");
        $this->readJsonProp($json, "basketErrors");
        $this->readJsonProp($json, "CLEARING_AMOUNT", 'amount');
        $this->readJsonProp($json, 'IDENTIFICATION_TRANSACTIONID', 'transactionid');
        $this->readJsonProp($json, 'IDENTIFICATION_UNIQUEID', 'uniqueid');
        $this->readJsonProp($json, 'IDENTIFICATION_SHORTID', 'shortid');
        $this->readJsonProp($json, 'PROCESSING_RESULT', 'result');
        $this->readJsonProp($json, 'PROCESSING_STATUS', 'status');
        $this->readJsonProp($json, 'PROCESSING_REASON', 'reason');
        $this->readJsonProp($json, 'PROCESSING_RETURN', 'return');
        $this->readJsonProp($json, 'PROCESSING_TIMESTAMP', 'timestamp');
        $this->readJsonProp($json, 'PRESENTATION_AMOUNT', 'amount');
        $this->readJsonProp($json, 'PRESENTATION_CURRENCY', 'currency');
        $this->readJsonProp($json, 'PAYMENT_CODE', 'paymentcode');
        $this->readJsonProp($json, 'PROCESSING_REASON_CODE', 'reasoncode');
        $this->readJsonProp($json, 'AUTHENTICATION_TYPE', 'authenticationtype');
        $this->readJsonProp($json, 'PROCESSING_RETURN_CODE', 'returncode');
        $this->readJsonProp($json, 'heidelpay_input_position', 'inputposition');
        $this->readJsonProp($json, 'PROCESSING_STATUS_CODE', 'statuscode');
        $this->readJsonProp($json, 'CONNECTOR_ACCOUNT_HOLDER', 'accountholder');
        $this->readJsonProp($json, 'ACCOUNT_BRAND', 'accountbrand');
        $this->readJsonProp($json, 'ACCOUNT_EXPIRY_MONTH', 'accountmonth');
        $this->readJsonProp($json, 'ACCOUNT_EXPIRY_YEAR', 'accountyear');
        $this->readJsonProp($json, 'CONNECTOR_ACCOUNT_BIC', 'accountbank');
        $this->readJsonProp($json, 'CONNECTOR_ACCOUNT_IBAN', 'accountnumber');
    }

    protected function readJsonProp($json, $property, $targetProperty = null)
    {
        if (isset($json->{$property})) {
            if (is_null($targetProperty)) {
                $targetProperty = $property;
            }
            $this->{$targetProperty} = $json->{$property};
        }
    }

    /**
     * Reads xml data and set properties
     *
     * @param       $oDomXPath
     * @param array $transactionData
     */
    protected function readXmlData(Domxpath $oDomXPath, array $transactionData = array())
    {
        if ($oDomXPath->query('/Response')->length) {
            $this->transactionid = $oDomXPath->query('/Response/Transaction/Identification/TransactionID')->item(
                0
            )->textContent;
            $this->uniqueid      = $oDomXPath->query('/Response/Transaction/Identification/UniqueID')->item(
                0
            )->textContent;
            $this->shortid       = $oDomXPath->query('/Response/Transaction/Identification/ShortID')->item(
                0
            )->textContent;
            $this->result        = $oDomXPath->query('/Response/Transaction/Processing/Result')->item(0)->textContent;
            $this->status        = $oDomXPath->query('/Response/Transaction/Processing/Status')->item(0)->textContent;
            $this->reason        = $oDomXPath->query('/Response/Transaction/Processing/Reason')->item(0)->textContent;
            $this->timestamp     = $oDomXPath->query('/Response/Transaction/Processing/Timestamp')->item(
                0
            )->textContent;
            $this->amount        = $oDomXPath->query('/Response/Transaction/Payment/Presentation/Amount')->item(
                0
            )->textContent;
            $this->currency      = $oDomXPath->query('/Response/Transaction/Payment/Presentation/Currency')->item(
                0
            )->textContent;

            $this->setPropertyViaXMLAttribute(
                $oDomXPath,
                '/Response/Transaction/Processing/Reason',
                'reasoncode',
                'code'
            );
            $this->setPropertyViaXMLAttribute(
                $oDomXPath,
                '/Response/Transaction/Processing/Return',
                'returncode',
                'code'
            );
            $this->setPropertyViaXMLAttribute($oDomXPath, '/Response/Transaction/Payment', 'paymentcode', 'code');
            $this->setPropertyViaXMLAttribute(
                $oDomXPath,
                '/Response/Transaction/Authentication',
                'authenticationtype',
                'type'
            );

            if ($oDomXPath->query('/Response/Transaction/Processing/Redirect')->item(0)) {
                $oRedirect         = $oDomXPath->query('/Response/Transaction/Processing/Redirect')->item(0);
                $this->redirecturl = $oRedirect->attributes->getNamedItem('url')->textContent;

                $this->setRedirectPropertiesViaXML($oDomXPath, $oRedirect);
            }

            if (isset($transactionData['heidelpay_input_position'])) {
                $this->inputposition = $transactionData['heidelpay_input_position'];
            }
        }
    }

    /**
     * sets property via xml attribute
     *
     * @param Domxpath     $oDomXPath
     * @param  string       $sQuery
     * @param  string       $sTargetProperty
     * @param        string $sAttributeName
     *
     * @return $this
     */
    protected function setPropertyViaXMLAttribute(Domxpath $oDomXPath, $sQuery, $sTargetProperty, $sAttributeName)
    {
        if ($oDomXPath->query($sQuery)->item(0)) {
            $this->$sTargetProperty = $oDomXPath->query($sQuery)->item(0)->attributes->getNamedItem(
                $sAttributeName
            )->textContent;
        }

        return $this;
    }

    /**
     * sets redirect properties via xml
     *
     * @param Domxpath $oDomXPath
     * @param           $oRedirect
     */
    protected function setRedirectPropertiesViaXML(Domxpath $oDomXPath, $oRedirect)
    {
        foreach ($oDomXPath->query('Parameter', $oRedirect) as $oNode) {
            $sName                            = $oNode->attributes->getNamedItem('name')->textContent;
            $this->redirectparameters[$sName] = $oNode->textContent;
        }
    }

    /**
     * reads array and set properties
     *
     * @param $transactionData
     */
    protected function readArray($transactionData)
    {
        $predefinedKeys = array(
            'IDENTIFICATION_TRANSACTIONID' => '',
            'IDENTIFICATION_UNIQUEID'      => '',
            'IDENTIFICATION_SHORTID'       => '',
            'PROCESSING_RESULT'            => '',
            'PROCESSING_STATUS'            => '',
            'PROCESSING_REASON'            => '',
            'PROCESSING_RETURN'            => '',
            'PROCESSING_TIMESTAMP'         => '',
            'PRESENTATION_AMOUNT'          => '',
            'PRESENTATION_CURRENCY'        => '',
            'PAYMENT_CODE'                 => '',
            'PROCESSING_REASON_CODE'       => '',
            'AUTHENTICATION_TYPE'          => '',
            'PROCESSING_RETURN_CODE'       => '',
            'heidelpay_input_position'     => '',
            'PROCESSING_STATUS_CODE'       => '',
            'ACCOUNT_BANK'                 => '',
            'ACCOUNT_NUMBER'               => '',
            'ACCOUNT_BIC'                  => '',
            'ACCOUNT_IBAN'                 => '',
            'ACCOUNT_HOLDER'               => '',
            'ACCOUNT_BRAND'                => '',
            'ACCOUNT_EXPIRY_MONTH'         => '',
            'ACCOUNT_EXPIRY_YEAR'          => '',
            'PROCESSING_REDIRECT_URL'      => '',
            'FRONTEND_PAYMENT_FRAME_URL'   => '',
        );

        $mergedData = array_merge($predefinedKeys, $transactionData);

        $this->transactionid      = $mergedData['IDENTIFICATION_TRANSACTIONID'];
        $this->uniqueid           = $mergedData['IDENTIFICATION_UNIQUEID'];
        $this->shortid            = $mergedData['IDENTIFICATION_SHORTID'];
        $this->result             = $mergedData['PROCESSING_RESULT'];
        $this->status             = $mergedData['PROCESSING_STATUS'];
        $this->reason             = $mergedData['PROCESSING_REASON'];
        $this->return             = $mergedData['PROCESSING_RETURN'];
        $this->timestamp          = $mergedData['PROCESSING_TIMESTAMP'];
        $this->amount             = $mergedData['PRESENTATION_AMOUNT'];
        $this->currency           = $mergedData['PRESENTATION_CURRENCY'];
        $this->paymentcode        = $mergedData['PAYMENT_CODE'];
        $this->reasoncode         = $mergedData['PROCESSING_REASON_CODE'];
        $this->authenticationtype = $mergedData['AUTHENTICATION_TYPE'];
        $this->returncode         = $mergedData['PROCESSING_RETURN_CODE'];
        $this->inputposition      = $mergedData['heidelpay_input_position'];
        $this->statuscode         = $mergedData['PROCESSING_STATUS_CODE'];
        $this->accountholder      = $mergedData['ACCOUNT_HOLDER'];
        $this->accountbrand       = $mergedData['ACCOUNT_BRAND'];
        $this->accountmonth       = $mergedData['ACCOUNT_EXPIRY_MONTH'];
        $this->accountyear        = $mergedData['ACCOUNT_EXPIRY_YEAR'];
        $this->accountbank        = $mergedData['ACCOUNT_BANK'];
        $this->accountnumber      = $mergedData['ACCOUNT_NUMBER'];

        if (false == empty($mergedData['ACCOUNT_IBAN'])) {
            $this->accountbank   = $mergedData['ACCOUNT_BIC'];
            $this->accountnumber = $mergedData['ACCOUNT_IBAN'];
        }

        if (false == empty($mergedData['PROCESSING_REDIRECT_URL'])) {
            $this->redirecturl = $mergedData['PROCESSING_REDIRECT_URL'];
        } elseif (false == empty($mergedData['FRONTEND_PAYMENT_FRAME_URL'])) {
            $this->redirecturl = $mergedData['FRONTEND_PAYMENT_FRAME_URL'];
        }

        foreach ($mergedData as $sKeyName => $sParameter) {
            if (false == substr_count($sKeyName, 'PROCESSING_REDIRECT_PARAMETER_')) {
                continue;
            }
            $sNewKeyName                            = str_replace('PROCESSING_REDIRECT_PARAMETER_', '', $sKeyName);
            $this->redirectparameters[$sNewKeyName] = $sParameter;
        }
    }

    /**
     * Return property redirectparameters
     *
     * @return array
     */
    public function getRedirectparameters()
    {
        return $this->redirectparameters;
    }

    /**
     * Return property amount
     *
     * @return string
     */
    public function getAmount()
    {
        return $this->amount;
    }

    /**
     * Return property currency
     *
     * @return string
     */
    public function getCurrency()
    {
        return $this->currency;
    }

    /**
     * Return property paymentcode
     *
     * @return string
     */
    public function getPaymentcode()
    {
        return $this->paymentcode;
    }

    /**
     * Return property reason
     *
     * @return string
     */
    public function getReason()
    {
        return $this->reason;
    }

    /**
     * Return property result
     *
     * @return string
     */
    public function getResult()
    {
        return $this->result;
    }

    /**
     * Return property shortid
     *
     * @return string
     */
    public function getShortid()
    {
        return $this->shortid;
    }

    /**
     * Return property status
     *
     * @return string
     */
    public function getStatus()
    {
        return $this->status;
    }

    /**
     * Return property timestamp
     *
     * @return string
     */
    public function getTimestamp()
    {
        return $this->timestamp;
    }

    /**
     * Return property transactionid
     *
     * @return string
     */
    public function getTransactionid()
    {
        return $this->transactionid;
    }

    /**
     * Return property uniqueid
     *
     * @return string
     */
    public function getUniqueid()
    {
        return $this->uniqueid;
    }

    /**
     * Return property returncode
     *
     * @return string
     */
    public function getReturncode()
    {
        return $this->returncode;
    }

    /**
     * Return property inputposition
     *
     * @return string
     */
    public function getInputposition()
    {
        return $this->inputposition;
    }

    /**
     * Return property authenticationtype
     *
     * @return string
     */
    public function getAuthenticationtype()
    {
        return $this->authenticationtype;
    }

    /**
     * Return property reasoncode
     *
     * @return mixed
     */
    public function getReasoncode()
    {
        return $this->reasoncode;
    }

    /**
     * Return property statuscode
     *
     * @return string
     */
    public function getStatuscode()
    {
        return $this->statuscode;
    }

    /**
     * Return property redirecturl
     *
     * @return string
     */
    public function getRedirecturl()
    {
        return $this->redirecturl;
    }

    /**
     * Return property accountbrand
     *
     * @return string
     */
    public function getAccountbrand()
    {
        return $this->accountbrand;
    }

    /**
     * Return property accountmonth
     *
     * @return string
     */
    public function getAccountmonth()
    {
        return $this->accountmonth;
    }

    /**
     * Return property accountyear
     *
     * @return string
     */
    public function getAccountyear()
    {
        return $this->accountyear;
    }

    /**
     * Return property accountholder
     *
     * @return string
     */
    public function getAccountholder()
    {
        return $this->accountholder;
    }

    /**
     * Return property accountnumber
     *
     * @return string
     */
    public function getAccountnumber()
    {
        return $this->accountnumber;
    }

    /**
     * Return property accountbank
     *
     * @return string
     */
    public function getAccountbank()
    {
        return $this->accountbank;
    }

    /**
     * @return mixed
     */
    public function getReturn()
    {
        return $this->return;
    }

    /**
     * @return array
     */
    public function getBasketErrors()
    {
        return $this->basketErrors;
    }

    /**
     * @return mixed
     */
    public function getBasketId()
    {
        return $this->basketId;
    }

    /**
     * @return mixed
     */
    public function getMethod()
    {
        return $this->method;
    }
}
