<?php

namespace d3\heidelpay\response;

use d3\heidelpay\transactionlog\reader\Transaction;
use stdClass;

/**
 * Class Parser
 */
class Parser
{
    /**
     * @var array
     */
    protected $responseData;

    /**#
     * @var Transaction
     */
    protected $transactiondata;

    /**
     * class constructor sets properties

*
*@param Transaction $oTransactiondata
     */
    public function __construct(Transaction $oTransactiondata)
    {
        $this->transactiondata = $oTransactiondata;
    }

    /**
     * Returns array of 3d secure data which is stored in the given transaction
     *
     * @return array
     */
    public function getThreeDSecureData()
    {
        $threeDSecureData                                 = array();
        $threeDSecureData['AUTHENTICATION_TYPE']          = $this->transactiondata->getAuthenticationtype();
        $threeDSecureData['PROCESSING_RESULT']            = $this->transactiondata->getResult();
        $threeDSecureData['PROCESSING_REASON_CODE']       = $this->transactiondata->getReasoncode();
        $threeDSecureData['PROCESSING_STATUS']            = $this->transactiondata->getStatus();
        $threeDSecureData['PROCESSING_REASON']            = $this->transactiondata->getReason();
        $threeDSecureData['PROCESSING_RETURN_CODE']       = $this->transactiondata->getReturncode();
        $threeDSecureData['IDENTIFICATION_SHORTID']       = $this->transactiondata->getShortid();
        $threeDSecureData['IDENTIFICATION_UNIQUEID']      = $this->transactiondata->getUniqueid();
        $threeDSecureData['IDENTIFICATION_TRANSACTIONID'] = $this->transactiondata->getTransactionid();
        $threeDSecureData['PAYMENT_CODE']                 = $this->transactiondata->getPaymentcode();

        return $threeDSecureData;
    }

    /**
     * Returns true if 3d secure is successful
     *
     * @return bool
     */
    public function isThreeDSecureSuccess()
    {
        return $this->isThreeDSercure() //
        && $this->transactiondata->getResult() == 'ACK' //
        && $this->transactiondata->getReasoncode() == '00';
    }

    /**
     * Returns true if transaction returns 3DSecure
     *
     * @return bool
     */
    public function isThreeDSercure()
    {
        return $this->transactiondata->getAuthenticationtype() == '3DSecure';
    }

    /**
     * Returns true if the transaction have a unique id and the result is ACK
     *
     * @return bool
     */
    public function isAckKnowledged()
    {
        return $this->transactiondata->getUniqueid() && $this->transactiondata->getResult() == 'ACK';
    }

    /**
     * Returns array of bank data which is extracted from given transaction
     *
     * @return array
     */
    public function getBankData()
    {
        $bankData                 = array();
        $bankData['lsbankname']   = ''; // kein Wert vorhanden
        $bankData['lsblz']        = $this->transactiondata->getAccountbank();
        $bankData['lsktoinhaber'] = $this->transactiondata->getAccountholder();
        $bankData['lsktonr']      = $this->transactiondata->getAccountnumber();

        return $bankData;
    }

    /**
     * Returns array of credit card data which is extracted from given transaction
     *
     * @return array
     */
    public function getCardData()
    {
        $cardData            = array();
        $cardData['kktype']  = $this->transactiondata->getAccountbrand();
        $cardData['kkmonth'] = $this->transactiondata->getAccountmonth();
        $cardData['kkyear']  = $this->transactiondata->getAccountyear();
        $cardData['kkname']  = $this->transactiondata->getAccountholder();
        $cardData['kkpruef'] = ''; // wird nicht von Transaction uebergeben :-)

        // das neue Heidelpaysystem uebergibt die KK-Nummer nun: 411111******1111
        $cardData['kknumber'] = $this->transactiondata->getAccountnumber();

        if (strlen($cardData['kknumber']) <= 4) {
            //von Transaction werden nur die letzten 4 Stellen zurueck gegeben. Drum die * davor.
            $cardData['kknumber'] = '**********' . $cardData['kknumber'];
        }

        return $cardData;
    }

    /**
     * Returns if 3d secure and status is waiting.
     *
     * @return bool
     */
    public function isWaiting()
    {
        return $this->transactiondata->getResult() == 'ACK' //
        && $this->transactiondata->getReasoncode() == '00' //
        && $this->transactiondata->getStatuscode() == '80';
    }

    /**
     * Returns stdClass with 3d secure redirect data
     *
     * @return stdClass
     */
    public function getThreeDSecureRedirectData()
    {
        $oThreeDSecure = new stdClass();

        $oThreeDSecure->sRedirectURL    = $this->transactiondata->getRedirecturl();
        $oThreeDSecure->aRedirectRarams = $this->transactiondata->getRedirectparameters();

        return $oThreeDSecure;
    }

    /**
     * Returns the current transaction data object.

*
*@return Transaction
     */
    public function getActualTransactionData()
    {
        return $this->transactiondata;
    }

    /**
     * Returns true if inputposition is otpayment (online transfer)
     *
     * @return bool
     */
    public function isOnlineTransferPaymentResponse()
    {
        return $this->transactiondata->getInputposition() == 'otpayment';
    }

    /**
     * Returns array of online transfer data
     *
     * @return array
     */
    public function getOnlineTransferData()
    {
        $onlineTransferData                                 = array();
        $onlineTransferData['PROCESSING_RESULT']            = $this->transactiondata->getResult();
        $onlineTransferData['PROCESSING_STATUS']            = $this->transactiondata->getStatus();
        $onlineTransferData['PROCESSING_REASON']            = $this->transactiondata->getReason();
        $onlineTransferData['IDENTIFICATION_SHORTID']       = $this->transactiondata->getShortid();
        $onlineTransferData['IDENTIFICATION_UNIQUEID']      = $this->transactiondata->getUniqueid();
        $onlineTransferData['IDENTIFICATION_TRANSACTIONID'] = $this->transactiondata->getTransactionid();
        $onlineTransferData['ACCOUNT_BRAND']                = $this->transactiondata->getAccountbrand();
        $onlineTransferData['PROCESSING_REASON_CODE']       = $this->transactiondata->getReasoncode();
        $onlineTransferData['PROCESSING_RETURN_CODE']       = $this->transactiondata->getReturncode();
        $onlineTransferData['PAYMENT_CODE']                 = $this->transactiondata->getPaymentcode();

        return $onlineTransferData;
    }

    /**
     * Returns true if online transfer is valid and successful
     *
     * @return bool
     */
    public function isOnlineTransferPaymentSuccess()
    {
        return $this->isSimpleSuccess()//
        && in_array($this->transactiondata->getPaymentcode(), array('OT.RC', 'IV.PA', 'VA.DB', 'VA.PA'));
    }

    /**
     * Returns true if transaction is basically successful
     *
     * @return bool
     */
    public function isSimpleSuccess()
    {
        return $this->transactiondata->getResult() == 'ACK' //
        && $this->transactiondata->getUniqueid() //
        && $this->transactiondata->getReturncode();
    }
}
